﻿//  Copyright 2012 ESRI

//  All rights reserved under the copyright laws of the United States.

//  You may freely redistribute and use this sample code, with or without modification.

//  Disclaimer: THE SAMPLE CODE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED
//  WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
//  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL ESRI OR
//  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
//  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
//  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//  INTERRUPTION) SUSTAINED BY YOU OR A THIRD PARTY, HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT ARISING IN ANY
//  WAY OUT OF THE USE OF THIS SAMPLE CODE, EVEN IF ADVISED OF THE POSSIBILITY OF
//  SUCH DAMAGE.

//  For additional information contact: Environmental Systems Research Institute, Inc.
//  Attn: Contracts Dept.

//  380 New York Street

//  Redlands, California, U.S.A. 92373

//  Email: contracts@esri.com

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ESRI.ArcGIS.Geodatabase;
using ESRI.ArcGIS.esriSystem;
using System.IO;
using System.Runtime.InteropServices;
using MongoDB.Driver;
using MongoDBPlugIn.Utilities;
using ESRI.ArcGIS.ADF.CATIDs;

namespace MongoDBPlugIn
{
  /// <summary>
  /// Provides starting point for the plug-in datasource architecture to consume our
  /// plug in helpers. Workspace helpers are created here
  /// </summary>
  [ClassInterface(ClassInterfaceType.None)]
  [Guid("3CC2CAC3-DDE8-4B66-80AB-E1FB0E49C88E")]
  [ProgId("MongoDBPlugIn.MongoDBWorkspacePluginFactory")]
  [ComVisible(true)]
  public class MongoDBWorkspacePluginFactory :
      IPlugInWorkspaceFactoryHelper
  {
    #region "Component Category Registration"

    [ComRegisterFunction()]
    public static void RegisterFunction(String regKey)
    {
      PlugInWorkspaceFactoryHelpers.Register(regKey);
    }

    [ComUnregisterFunction()]
    public static void UnregisterFunction(String regKey)
    {
      PlugInWorkspaceFactoryHelpers.Unregister(regKey);
    }
    #endregion

    /// <summary>
    /// Attempts opening a MongoDBWorkspace from its connection file path
    /// </summary>
    /// <param name="wksString">the path to a connection file</param>
    /// <returns>a mongodbworkspace</returns>
    public MongoDBWorkspace OpenMongoDBWorkspace(string wksString)
    {
      MongoDatabase mDB;
      try
      {
        string connString = ConnectionUtilities.DecodeConnFile(wksString);
        MongoDBConnInfo connInfo = ConnectionUtilities.ParseConnectionString(connString);
        mDB = ConnectionUtilities.OpenConnection(connInfo);
      }
      catch (Exception e)
      {
        throw new COMException(e.Message);
      }
      return new MongoDBWorkspace(mDB);
    }

    #region IPlugInWorkspaceFactoryHelper methods
    /// <summary>
    /// whether we support SQL - currently no
    /// </summary>
    public bool CanSupportSQL
    {
      get
      {
        return false;
      }
    }

    /// <summary>
    /// whether a particular directory contains a workspace
    /// </summary>
    /// <param name="parentDirectory">the directory to check</param>
    /// <param name="fileNames">contained files</param>
    /// <returns>whether it contains a workspace</returns>
    public bool ContainsWorkspace(string parentDirectory, IFileNames fileNames)
    {
      return this.IsWorkspace(parentDirectory);
    }

    /// <summary>
    /// name of the datasource
    /// Note: this progid "esriGeoDatabase.[DatasourceName]WorkspaceFactory"
    /// Is used to create the workspace factory wrapping this plug in helper
    /// </summary>
    public string DataSourceName
    {
      get
      {
        return "MongoDBPlugin";
      }
    }

    /// <summary>
    /// Gets a workspace string representing the workspace
    /// </summary>
    /// <param name="parentDirectory">directory</param>
    /// <param name="fileNames">files</param>
    /// <returns>string</returns>
    public string GetWorkspaceString(string parentDirectory, IFileNames fileNames)
    {
      if (this.IsWorkspace(parentDirectory))
        return parentDirectory;

      return null;
    }
    
    /// <summary>
    /// Determins whether a string represents a workspace
    /// Note that workspace strings are file paths to conn files in this plugin
    /// </summary>
    /// <param name="wksString">the string (i.e. path to conn file)</param>
    /// <returns>whether it is or not</returns>
    public bool IsWorkspace(string wksString)
    {
      bool retVal = false;
      try
      {
        string connString = ConnectionUtilities.DecodeConnFile(wksString);
        MongoDBConnInfo connInfo = ConnectionUtilities.ParseConnectionString(connString);
        ConnectionUtilities.OpenConnection(connInfo);
        retVal = true;
      }
      catch (Exception)
      {
        // this just means this isn't a proper string
      }

      return retVal;
    }

    /// <summary>
    /// Opens a IPlugInWorkspaceHelper for a particualar connection file
    /// </summary>
    /// <param name="wksString">connection file path</param>
    /// <returns>IPlugInWorkspaceHelper</returns>
    public IPlugInWorkspaceHelper OpenWorkspace(string wksString)
    {
      return (IPlugInWorkspaceHelper)OpenMongoDBWorkspace(wksString);
    }

    /// <summary>
    /// A mock UID for the factory - necessary for the plug-in factory to 
    /// masquerade as an ordianry workspace factory
    /// </summary>
    public UID WorkspaceFactoryTypeID
    {
      get
      {
        UID proxyUID = new UIDClass();
        proxyUID.Value = "{11EDE71D-928E-4546-B51F-E4A343C32D9E}";
        return proxyUID;
      }
    }

    /// <summary>
    /// the sort of workspace. As we are storing conn info in 
    /// conn files, this is file system
    /// </summary>
    public esriWorkspaceType WorkspaceType
    {
      get
      {
        return esriWorkspaceType.esriRemoteDatabaseWorkspace;
      }
    }

    /// <summary>
    /// Gets names of various datasets
    /// note we only support feature classes currently
    /// </summary>
    /// <param name="DatasetType"></param>
    /// <returns></returns>
    public string get_DatasetDescription(esriDatasetType DatasetType)
    {
      switch (DatasetType)
      {
        case esriDatasetType.esriDTFeatureClass:
          return "MongoDB Feature Class";
        default:
          return null;
      }
    }

    /// <summary>
    /// Gets a description of the workspace
    /// </summary>
    /// <param name="plural">Whether we want the plural or ordinary form</param>
    /// <returns>string</returns>
    public string get_WorkspaceDescription(bool plural)
    {
      if (plural)
        return "MongoDB repostories";
      else
        return "MongoDB repository";
    }
    #endregion

  }
}
